# LICENSE: You're free to distribute this under the same terms as Perl itself.

use strict;
use Carp ();
use LWP::UserAgent;

############################################################################
package Net::OpenID::Consumer;

use vars qw($VERSION $HAS_CRYPT_DSA $HAS_CRYPT_OPENSSL $HAS_OPENSSL);
$VERSION = "0.01";

use fields (
            'cacher',         # the Net::OpenID::Cacher::* class to remember mapping of OpenID -> Identity Server
            'ua',             # LWP::UserAgent instance to use
            'server_selector',# optional subref that will pick which identity server to use, if multiple 
            'last_errcode', # last error code we got
            'last_errtext', # last error code we got
            );

use Net::OpenID::ClaimedIdentity;
use Net::OpenID::VerifiedIdentity;
use MIME::Base64 ();
use Digest::SHA1 ();

BEGIN {
    unless ($HAS_CRYPT_OPENSSL = eval "die; use Crypt::OpenSSL::DSA (); 1;") {
        unless ($HAS_CRYPT_DSA = eval "die 'FIXME_BELOW'; use Crypt::DSA (); use Convert::PEM; 1;") {
            unless ($HAS_OPENSSL = `which openssl`) {
                die "Net::OpenID::Consumer failed to load, due to missing dependencies.  You to have Crypt::OpenSSL::DSA -or- the binary 'openssl' in your path.";
            }
        }
    }
}

sub new {
    my Net::OpenID::Consumer $self = shift;
    $self = fields::new( $self ) unless ref $self;
    $self->{cacher} = undef;
    $self->{ua} = undef;
    $self->{last_errcode} = undef;
    $self->{last_errtext} = undef;
    return $self;
}

sub cacher {
    my Net::OpenID::Consumer $self = shift;
    $self->{cacher} = shift if @_;
    $self->{cacher};
}

sub server_selector {
    my Net::OpenID::Consumer $self = shift;
    if (@_) {
        my $code = shift;
        Carp::croak("Not a CODE ref") unless ref $code eq "CODE";
        $self->{server_selector} = $code;
    }
    $self->{server_selector};
}

sub ua {
    my Net::OpenID::Consumer $self = shift;
    $self->{ua} = shift if @_;

    # make default one on first access
    unless ($self->{ua}) {
        my $ua = $self->{ua} = LWP::UserAgent->new;
        $ua->timeout(10);
    }

    $self->{ua};
}

sub _fail {
    my Net::OpenID::Consumer $self = shift;
    $self->{last_errcode} = shift;
    $self->{last_errtext} = shift;
    wantarray ? () : undef;
}

sub json_err {
    my Net::OpenID::Consumer $self = shift;
    return OpenID::util::js_dumper({
        err_code => $self->{last_errcode},
        err_text => $self->{last_errtext},
    });
}

sub err {
    my Net::OpenID::Consumer $self = shift;
    $self->{last_errcode};
}

sub errtext {
    my Net::OpenID::Consumer $self = shift;
    $self->{last_errtext};
}

sub get_url_contents {
    my Net::OpenID::Consumer $self = shift;
    my $url = shift;
    my $final_url_ref = shift;

    # FIXME: use cacher

    my $res = $self->ua->get($url);
    if ($res->is_success) {
        $$final_url_ref = $res->request->uri->as_string;
        return $res->content;
    }
    return $self->_fail("url_fetch_error", "Error fetching URL: " . $res->status_line);
}

sub pick_identity_server {
    my Net::OpenID::Consumer $self = shift;
    my $id_server_list = shift;

    if (my $hook = $self->{server_selector}) {
        return $hook->($self, $id_server_list);
    }

    # default just picks first one.
    return $id_server_list->[0];
}

sub find_openid_servers {
    my Net::OpenID::Consumer $self = shift;
    my $url = shift;
    my $final_url_ref = shift;

    my $doc = $self->get_url_contents($url, $final_url_ref) or
        return;

    my @id_servers;
    while ($doc =~ m!<link([^>]+)>!g) {
        my $link = $1;
        if ($link =~ /rel=.openid\.server./i &&
            $link =~ m!href=[\"\']([^\"\']+)[\"\']!i) {
            push @id_servers, $1;
        }
    }

    return $self->_fail("no_identity_servers") unless @id_servers;
    @id_servers;
}

# returns Net::OpenID::ClaimedIdentity
sub get_claimed_identity {
    my Net::OpenID::Consumer $self = shift;
    my ($url) = @_;

    # trim whitespace
    $url =~ s/^\s+//;
    $url =~ s/\s+$//;
    return helper_error("empty_url", "Empty URL") unless $url;

    # do basic canonicalization
    $url = "http://$url" if $url && $url !~ m!^\w+://!;
    return helper_error("bogus_url", "Invalid URL") unless $url =~ m!^http://!;
    # add a slash, if none exists
    $url .= "/" unless $url =~ m!^http://.+/!;

    my $final_url;
    my @id_servers = $self->find_openid_servers($url, \$final_url)
        or return;

    return Net::OpenID::ClaimedIdentity->new(
                                             identity => $final_url,
                                             servers => \@id_servers,
                                             consumer => $self,
                                             );
}

# given something that can have GET arguments, returns a subref to get them:
#   CGI.pm object
#   hashref of GET args
#   modperl $r
#   ...
sub _param_getter {
    my $what = shift;
    if (ref $what eq "HASH") {
        return sub { $what->{$_[0]}; };
    } else {
        Carp::croak("Unknown parameter type ($what)");
    }
}

sub user_setup_url {
    my Net::OpenID::Consumer $self = shift;
    my $param = _param_getter(shift);
    return $self->_fail("bad_mode") unless $param->("openid.mode") eq "id_res";
    return $param->("openid.user_setup_url");
}

sub verified_identity {
    my Net::OpenID::Consumer $self = shift;
    my $param = _param_getter(shift);
    return $self->_fail("bad_mode") unless $param->("openid.mode") eq "id_res";

    my $sig64 = $param->("openid.sig")             or return $self->_fail("no_sig");
    my $url   = $param->("openid.assert_identity") or return $self->_fail("no_identity");
    my $retto = $param->("openid.return_to")       or return $self->_fail("no_return_to");

    # present and valid
    my $ts  = $param->("openid.timestamp");
    $ts =~ /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}Z$/ or return $self->_fail("malformed_timestamp");

    # make the raw string that we're going to check the signature against
    my $msg_plain = join("::",
                         $ts,
                         "assert_identity",
                         $url,
                         $retto);

    # to verify the signature, we need to fetch the public key, which
    # means we need to figure out what identity server to get the public
    # key from.  because there might be multiple, we'd previously
    # passed to ourselves the index that we chose.  so first go
    # re-fetch (possibly from cache) the page, re-find the acceptable
    # identity servers for this user, and get the public key
    my $final_url;
    my @id_servers = $self->find_openid_servers($url, \$final_url)
        or return undef;

    return $self->_fail("identity_changed_on_fetch")
        if $url ne $final_url;

    my $used_idx = int($param->("oicsr.idx") || 0);
    return $self->_fail("bad_idx")
        if $used_idx < 0 || $used_idx > 50;

    my $id_server = $id_servers[$used_idx]
        or return $self->_fail("identity_server_idx_empty");

    my $pem_url = $id_server;
    $pem_url .= ($id_server =~ /\?/) ? "&" : "?";
    $pem_url .= "openid.mode=getpubkey";

    my $msg = Digest::SHA1::sha1($msg_plain);
    my $sig = MIME::Base64::decode_base64($sig64);

    # TODO: foreach my $mode ("cached", "no_cache")
    my $public_pem = $self->get_url_contents($pem_url)
        or return $self->_fail("public_key_fetch_error", "Details: " . $self->err . ": " . $self->errtext);

    $self->_dsa_verify($public_pem, $sig, $msg, $msg_plain)
        or return undef;

    # FIXME: nonce callback
    return Net::OpenID::VerifiedIdentity->new(
                                              identity => $url,
                                              );
}

sub _dsa_verify {
    my ($self, $public_pem, $sig, $msg, $msg_plain) = @_;

    if ($HAS_CRYPT_OPENSSL) {
        my $dsa_pub  = Crypt::OpenSSL::DSA->read_pub_key_str($public_pem)
            or $self->_fail("pubkey_parse_error", "Couldn't parse public key");
        $dsa_pub->verify($msg, $sig)
            or return $self->_fail("verify_failed", "DSA signature verification failed");
        return 1;
    }

    if ($HAS_CRYPT_DSA) {
        my $cd = Crypt::DSA->new;

        my ($len, $len_r, $len_s, $r, $s);
        unless ($sig =~ /^\x30/ &&
                ($len = ord(substr($sig,1,1))) &&
                substr($sig,2,1) eq "\x02" &&
                ($len_r =  ord(substr($sig,3,1))) &&
                ($r = substr($sig,4,$len_r)) &&
                substr($sig,4+$len_r,1) eq "\x02" &&
                ($len_s =  ord(substr($sig,5+$len_r,1))) &&
                ($s = substr($sig,6+$len_r,$len_s))) {
            return $self->_fail("asn1_parse_error", "Failed to parse ASN.1 signature");
        }

        my $sigobj = Crypt::DSA::Signature->new;
        $sigobj->r("0x" . unpack("H40", $r));
        $sigobj->s("0x" . unpack("H40", $s));


        die "#### FIXME: Crypt::DSA::Key only parses private keys.  Need to fix it.";

        my $key =  Crypt::DSA::Key->new(
                                        Type => "PEM",
                                        Content => $public_pem,
                                        )
            or return $self->_fail("pubkey_parse_error", "Couldn't generate Crypt::DSA::Key from PEM");

        $cd->verify(
                    Digest    => $msg,
                    Signature => $sigobj,
                    Key       => $key,
                    )
            or return $self->_fail("verify_failed", "DSA signature verification failed");
        return 1;
    }

    if ($HAS_OPENSSL) {
        require File::Temp;
        my $sig_temp = new File::Temp(TEMPLATE => "tmp.signatureXXXX") or die;
        my $pub_temp = new File::Temp(TEMPLATE => "tmp.pubkeyXXXX") or die;
        my $msg_temp = new File::Temp(TEMPLATE => "tmp.msgXXXX") or die;
        syswrite($sig_temp,$sig);
        syswrite($pub_temp,$public_pem);
        syswrite($msg_temp,$msg_plain);
        # FIXME: shutup openssl from spewing to STDOUT the "Verification OK".
        my $rv = system("openssl", "dgst", "-dss1", "-verify", "$pub_temp", "-signature", "$sig_temp", "$msg_temp");
        return $self->_fail("verify_failed", "DSA signature verification failed") if $rv;
        return 1;
    }

    return 0;
}

package OpenID::util;

sub ejs
{
    my $a = $_[0];
    $a =~ s/[\"\'\\]/\\$&/g;
    $a =~ s/\r?\n/\\n/gs;
    $a =~ s/\r//;
    return $a;
}

# Data::Dumper for JavaScript
sub js_dumper {
    my $obj = shift;
    if (ref $obj eq "HASH") {
        my $ret = "{";
        foreach my $k (keys %$obj) {
            $ret .= "$k: " . js_dumper($obj->{$k}) . ",";
        }
        chop $ret;
        $ret .= "}";
        return $ret;
    } elsif (ref $obj eq "ARRAY") {
        my $ret = "[" . join(", ", map { js_dumper($_) } @$obj) . "]";
        return $ret;
    } else {
        return $obj if $obj =~ /^\d+$/;
        return "\"" . ejs($obj) . "\"";
    }
}

sub eurl
{
    my $a = $_[0];
    $a =~ s/([^a-zA-Z0-9_\,\-.\/\\\: ])/uc sprintf("%%%02x",ord($1))/eg;
    $a =~ tr/ /+/;
    return $a;
}
